/*
 * Copyright (c) 2022-2024 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#if defined(ARM_COMPUTE_ENABLE_SME2)

#include "arm_gemm.hpp"
#include "../../utils.hpp"
#include "../../bfloat.hpp"

#include <cassert>
#include <limits>

namespace arm_gemm {

void sme2_gemv_fp32bf16fp32_dot_16VL (
    const float *A_ptr, const bfloat16 *B_ptr, float *output_ptr,
    size_t N, size_t K,
    const float *bias, Activation act, bool
)
{
    struct KernelArgs {
        float maxval = static_cast<float>(std::numeric_limits<float>::infinity());
        float minval = - static_cast<float>(std::numeric_limits<float>::infinity());
        const bfloat16 *B_ptr = {};
        size_t output_offset = {};
        unsigned int input_initial_col = {};
    } ka;

    unsigned long flags=0;
    ka.B_ptr = B_ptr;
    switch(act.type) {
        default:
        case Activation::Type::None:
            break;
        case Activation::Type::BoundedReLU:
            ka.maxval = static_cast<float>(act.param1);
            /* fall through */
        case Activation::Type::ReLU:
            ka.minval = 0;
            flags |= 0x2;
            break;
    }
    __asm__ __volatile__(
      "ptrue p8.b\n"
      ".inst 0xd503477f  // SMSTART ZA\n"
      "mov x10, #0x0\n"
      "cntw x9, ALL, MUL #4\n"
      "mov x28, #0x4\n"
      "add x27, %x[N], x9\n"
      "mov x26, %x[B_ptr]\n"
      "sub x27, x27, #0x1\n"
      "mov x25, %x[output_ptr]\n"
      "udiv x27, x27, x9\n"
      "ptrue p2.b\n"
      "add x22, x27, #0x3\n"
      ".inst 0x25207811  // ptrue pn9.b\n"
      "and x22, x22, #0xfffffffffffffffc\n"
      "mov x21, #0x1\n"
      "mul x22, x22, x9\n"
      "mul x22, x22, %x[K]\n"
      "lsl x22, x22, #0x1\n"
      "1:"  // RHS size check loop
      "cmp x22, #0x200000\n"
      "blt 2f\n"
      "tbnz x22, #0, 3f\n"
      "lsr x22, x22, #0x1\n"
      "lsl x21, x21, #0x1\n"
      "b 1b\n"
      "2:"  // RHS do prefetch
      "lsl x20, x22, #0x26\n"
      "sub x21, x21, #0x1\n"
      "lsl x21, x21, #0x16\n"
      "orr x22, x22, x20\n"
      "orr x22, x22, x21\n"
      ".inst 0xf8b64b5a  // rprfm pldonce, x22, [x26]\n"
      "3:"  // RHS prefetch exit
      "mov x24, %x[bias]\n"
      "4:"  // Column loop
      "cmp x27, #0x4\n"
      "bge 28f\n"
      "cmp x27, #0x2\n"
      "bgt 20f\n"
      "beq 12f\n"
      "mov x23, %x[A_ptr]\n"
      "lsl x21, %x[K], #0x2\n"
      "mov x20, %x[N]\n"
      "mov x22, %x[K]\n"
      ".inst 0xf8b54af8  // rprfm pldmany, x21, [x23]\n"
      ".inst 0x25b467f0  // whilelt p8.s, XZR, x20, VLx4\n"
      "cbz x24, 5f\n"
      ".inst 0xa040c710  // ld1w { z16.s-z19.s }, pn9.b/Z, [x24]\n"
      ".inst 0xc0044e00  // mova za.d[x10, #0], { z16.d-z19.d }\n"
      "b 6f\n"
      "5:"  // Width 1: no bias
      ".inst 0xc00800ff  // zero { zad0, zad1, zad2, zad3, zad4, zad5, zad6, zad7 }\n"
      "6:"  // Width 1: setup done
      "cmp x22, #0x8\n"
      "ble 8f\n"
      "7:"  // Width 1: Multiply loop: Main loop head
      "whilelt p1.s, XZR, x22\n"
      "whilelt p0.s, x28, x22\n"
      ".inst 0xa040a745  // ldnt1h { z4.h-z7.h }, pn9.b/Z, [x26]\n"
      "ld1rqw { z8.s }, p1/Z, [x23]\n"
      "addvl x26, x26, #16\n"
      "sub x22, x22, #0x8\n"
      "ld1rqw { z11.s }, p0/Z, [x23, #16]\n"
      "cmp x22, #0x8\n"
      "add x23, x23, #0x20\n"
      ".inst 0xa040a755  // ldnt1h { z20.h-z23.h }, pn9.b/Z, [x26]\n"
      "addvl x26, x26, #16\n"
      ".inst 0x658aa908  // bfcvt z8.h, p2/M, z8.s\n"
      ".inst 0xa040a751  // ldnt1h { z16.h-z19.h }, pn9.b/Z, [x26]\n"
      "addvl x26, x26, #16\n"
      ".inst 0x658aa96b  // bfcvt z11.h, p2/M, z11.s\n"
      ".inst 0xa040a74d  // ldnt1h { z12.h-z15.h }, pn9.b/Z, [x26]\n"
      "addvl x26, x26, #16\n"
      "uzp1 z8.h, z8.h, z8.h\n"
      "uzp1 z11.h, z11.h, z11.h\n"
      "trn1 z8.d, z8.d, z11.d\n"
      ".inst 0xc158d098  // bfdot za.s[x10, 0], { z4.h-z7.h }, z8.h[0]\n"
      ".inst 0xc158d698  // bfdot za.s[x10, 0], { z20.h-z23.h }, z8.h[1]\n"
      ".inst 0xc158da18  // bfdot za.s[x10, 0], { z16.h-z19.h }, z8.h[2]\n"
      ".inst 0xc158dd98  // bfdot za.s[x10, 0], { z12.h-z15.h }, z8.h[3]\n"
      "bgt 7b\n"
      "8:"  // Width 1: Multiply loop: Single iteration only
      "whilelt p1.s, XZR, x22\n"
      "whilelt p0.s, x28, x22\n"
      ".inst 0xa040a74d  // ldnt1h { z12.h-z15.h }, pn9.b/Z, [x26]\n"
      "ld1rqw { z3.s }, p1/Z, [x23]\n"
      "subs x22, x22, #0x2\n"
      "addvl x26, x26, #16\n"
      "ld1rqw { z24.s }, p0/Z, [x23, #16]\n"
      "add x23, x23, #0x20\n"
      ".inst 0x658aa863  // bfcvt z3.h, p2/M, z3.s\n"
      ".inst 0x658aab18  // bfcvt z24.h, p2/M, z24.s\n"
      "uzp1 z3.h, z3.h, z3.h\n"
      "uzp1 z24.h, z24.h, z24.h\n"
      "trn1 z3.d, z3.d, z24.d\n"
      ".inst 0xc153d198  // bfdot za.s[x10, 0], { z12.h-z15.h }, z3.h[0]\n"
      "ble 9f\n"
      ".inst 0xa040a74d  // ldnt1h { z12.h-z15.h }, pn9.b/Z, [x26]\n"
      "subs x22, x22, #0x2\n"
      "addvl x26, x26, #16\n"
      ".inst 0xc153d598  // bfdot za.s[x10, 0], { z12.h-z15.h }, z3.h[1]\n"
      "ble 9f\n"
      ".inst 0xa040a74d  // ldnt1h { z12.h-z15.h }, pn9.b/Z, [x26]\n"
      "subs x22, x22, #0x2\n"
      "addvl x26, x26, #16\n"
      ".inst 0xc153d998  // bfdot za.s[x10, 0], { z12.h-z15.h }, z3.h[2]\n"
      "ble 9f\n"
      ".inst 0xa040a75d  // ldnt1h { z28.h-z31.h }, pn9.b/Z, [x26]\n"
      "addvl x26, x26, #16\n"
      ".inst 0xc153df98  // bfdot za.s[x10, 0], { z28.h-z31.h }, z3.h[3]\n"
      "9:"  // Width 1: Multiply loop: multiply skip
      "tbz %x[flags], #1, 10f\n"
      "add x21, %x[args_ptr], %[offset_min]\n"
      "add x20, %x[args_ptr], %[offset_max]\n"
      ".inst 0xc0064c1c  // mova { z28.d-z31.d }, za.d[x10, #0]\n"
      "ld1rw { z4.s }, p2/Z, [x21]\n"
      "ld1rw { z18.s }, p2/Z, [x20]\n"
      ".inst 0xc1b2c89c  // fclamp { z28.s-z31.s }, z4.s, z18.s\n"
      ".inst 0xa060c33c  // st1w { z28.s-z31.s }, p8, [x25]\n"
      "addvl x25, x25, #4\n"
      "b 11f\n"
      "10:"  // Width 1: No activation
      ".inst 0xc0064c04  // mova { z4.d-z7.d }, za.d[x10, #0]\n"
      ".inst 0xa060c324  // st1w { z4.s-z7.s }, p8, [x25]\n"
      "addvl x25, x25, #4\n"
      "11:"  // Width 1: Output done
      "b 36f\n"
      "12:"  // Width 2
      "mov x23, %x[A_ptr]\n"
      "lsl x21, %x[K], #0x2\n"
      "sub x20, %x[N], x9\n"
      "mov x22, %x[K]\n"
      ".inst 0xf8b54af8  // rprfm pldmany, x21, [x23]\n"
      ".inst 0x25b467f0  // whilelt p8.s, XZR, x20, VLx4\n"
      "cbz x24, 13f\n"
      ".inst 0xa040c710  // ld1w { z16.s-z19.s }, pn9.b/Z, [x24]\n"
      ".inst 0xa041c700  // ld1w { z0.s-z3.s }, pn9.b/Z, [x24, #0x4, MUL VL]\n"
      ".inst 0xc0044e00  // mova za.d[x10, #0], { z16.d-z19.d }\n"
      ".inst 0xc0044c01  // mova za.d[x10, #1], { z0.d-z3.d }\n"
      "b 14f\n"
      "13:"  // Width 2: no bias
      ".inst 0xc00800ff  // zero { zad0, zad1, zad2, zad3, zad4, zad5, zad6, zad7 }\n"
      "14:"  // Width 2: setup done
      "cmp x22, #0x8\n"
      "ble 16f\n"
      "15:"  // Width 2: Multiply loop: Main loop head
      "whilelt p1.s, XZR, x22\n"
      "whilelt p0.s, x28, x22\n"
      ".inst 0xa040a755  // ldnt1h { z20.h-z23.h }, pn9.b/Z, [x26]\n"
      "ld1rqw { z8.s }, p1/Z, [x23]\n"
      "sub x22, x22, #0x8\n"
      "ld1rqw { z9.s }, p0/Z, [x23, #16]\n"
      "cmp x22, #0x8\n"
      "add x23, x23, #0x20\n"
      ".inst 0xa041a759  // ldnt1h { z24.h-z27.h }, pn9.b/Z, [x26, #0x4, MUL VL]\n"
      "addvl x26, x26, #16\n"
      ".inst 0x658aa908  // bfcvt z8.h, p2/M, z8.s\n"
      ".inst 0xa040a741  // ldnt1h { z0.h-z3.h }, pn9.b/Z, [x26]\n"
      ".inst 0x658aa929  // bfcvt z9.h, p2/M, z9.s\n"
      ".inst 0xa041a751  // ldnt1h { z16.h-z19.h }, pn9.b/Z, [x26, #0x4, MUL VL]\n"
      "addvl x26, x26, #16\n"
      ".inst 0xa040a745  // ldnt1h { z4.h-z7.h }, pn9.b/Z, [x26]\n"
      ".inst 0xa041a74d  // ldnt1h { z12.h-z15.h }, pn9.b/Z, [x26, #0x4, MUL VL]\n"
      "addvl x26, x26, #16\n"
      "uzp1 z8.h, z8.h, z8.h\n"
      ".inst 0xa040a75d  // ldnt1h { z28.h-z31.h }, pn9.b/Z, [x26]\n"
      "uzp1 z9.h, z9.h, z9.h\n"
      "trn1 z8.d, z8.d, z9.d\n"
      ".inst 0xc158d298  // bfdot za.s[x10, 0], { z20.h-z23.h }, z8.h[0]\n"
      ".inst 0xa041a755  // ldnt1h { z20.h-z23.h }, pn9.b/Z, [x26, #0x4, MUL VL]\n"
      "addvl x26, x26, #16\n"
      ".inst 0xc158d319  // bfdot za.s[x10, 1], { z24.h-z27.h }, z8.h[0]\n"
      ".inst 0xc158d418  // bfdot za.s[x10, 0], { z0.h-z3.h }, z8.h[1]\n"
      ".inst 0xc158d619  // bfdot za.s[x10, 1], { z16.h-z19.h }, z8.h[1]\n"
      ".inst 0xc158d898  // bfdot za.s[x10, 0], { z4.h-z7.h }, z8.h[2]\n"
      ".inst 0xc158d999  // bfdot za.s[x10, 1], { z12.h-z15.h }, z8.h[2]\n"
      ".inst 0xc158df98  // bfdot za.s[x10, 0], { z28.h-z31.h }, z8.h[3]\n"
      ".inst 0xc158de99  // bfdot za.s[x10, 1], { z20.h-z23.h }, z8.h[3]\n"
      "bgt 15b\n"
      "16:"  // Width 2: Multiply loop: Single iteration only
      "whilelt p1.s, XZR, x22\n"
      "whilelt p0.s, x28, x22\n"
      ".inst 0xa040a759  // ldnt1h { z24.h-z27.h }, pn9.b/Z, [x26]\n"
      "ld1rqw { z3.s }, p1/Z, [x23]\n"
      "subs x22, x22, #0x2\n"
      "ld1rqw { z16.s }, p0/Z, [x23, #16]\n"
      "add x23, x23, #0x20\n"
      ".inst 0xa041a745  // ldnt1h { z4.h-z7.h }, pn9.b/Z, [x26, #0x4, MUL VL]\n"
      "addvl x26, x26, #16\n"
      ".inst 0x658aa863  // bfcvt z3.h, p2/M, z3.s\n"
      ".inst 0x658aaa10  // bfcvt z16.h, p2/M, z16.s\n"
      "uzp1 z3.h, z3.h, z3.h\n"
      "uzp1 z16.h, z16.h, z16.h\n"
      "trn1 z3.d, z3.d, z16.d\n"
      ".inst 0xc153d318  // bfdot za.s[x10, 0], { z24.h-z27.h }, z3.h[0]\n"
      ".inst 0xc153d099  // bfdot za.s[x10, 1], { z4.h-z7.h }, z3.h[0]\n"
      "ble 17f\n"
      ".inst 0xa040a745  // ldnt1h { z4.h-z7.h }, pn9.b/Z, [x26]\n"
      "subs x22, x22, #0x2\n"
      ".inst 0xa041a759  // ldnt1h { z24.h-z27.h }, pn9.b/Z, [x26, #0x4, MUL VL]\n"
      "addvl x26, x26, #16\n"
      ".inst 0xc153d498  // bfdot za.s[x10, 0], { z4.h-z7.h }, z3.h[1]\n"
      ".inst 0xc153d719  // bfdot za.s[x10, 1], { z24.h-z27.h }, z3.h[1]\n"
      "ble 17f\n"
      ".inst 0xa040a745  // ldnt1h { z4.h-z7.h }, pn9.b/Z, [x26]\n"
      "subs x22, x22, #0x2\n"
      ".inst 0xa041a74d  // ldnt1h { z12.h-z15.h }, pn9.b/Z, [x26, #0x4, MUL VL]\n"
      "addvl x26, x26, #16\n"
      ".inst 0xc153d898  // bfdot za.s[x10, 0], { z4.h-z7.h }, z3.h[2]\n"
      ".inst 0xc153d999  // bfdot za.s[x10, 1], { z12.h-z15.h }, z3.h[2]\n"
      "ble 17f\n"
      ".inst 0xa040a745  // ldnt1h { z4.h-z7.h }, pn9.b/Z, [x26]\n"
      ".inst 0xa041a755  // ldnt1h { z20.h-z23.h }, pn9.b/Z, [x26, #0x4, MUL VL]\n"
      "addvl x26, x26, #16\n"
      ".inst 0xc153dc98  // bfdot za.s[x10, 0], { z4.h-z7.h }, z3.h[3]\n"
      ".inst 0xc153de99  // bfdot za.s[x10, 1], { z20.h-z23.h }, z3.h[3]\n"
      "17:"  // Width 2: Multiply loop: multiply skip
      "tbz %x[flags], #1, 18f\n"
      "add x21, %x[args_ptr], %[offset_min]\n"
      "add x20, %x[args_ptr], %[offset_max]\n"
      ".inst 0xc0064c08  // mova { z8.d-z11.d }, za.d[x10, #0]\n"
      ".inst 0xc0064c2c  // mova { z12.d-z15.d }, za.d[x10, #1]\n"
      "ld1rw { z3.s }, p2/Z, [x21]\n"
      "ld1rw { z28.s }, p2/Z, [x20]\n"
      ".inst 0xc1bcc868  // fclamp { z8.s-z11.s }, z3.s, z28.s\n"
      ".inst 0xc1bcc86c  // fclamp { z12.s-z15.s }, z3.s, z28.s\n"
      ".inst 0xa060c728  // st1w { z8.s-z11.s }, pn9.b, [x25]\n"
      ".inst 0xa061c32c  // st1w { z12.s-z15.s }, p8, [x25, #0x4, MUL VL]\n"
      "addvl x25, x25, #8\n"
      "b 19f\n"
      "18:"  // Width 2: No activation
      ".inst 0xc0064c04  // mova { z4.d-z7.d }, za.d[x10, #0]\n"
      ".inst 0xc0064c38  // mova { z24.d-z27.d }, za.d[x10, #1]\n"
      ".inst 0xa060c724  // st1w { z4.s-z7.s }, pn9.b, [x25]\n"
      ".inst 0xa061c338  // st1w { z24.s-z27.s }, p8, [x25, #0x4, MUL VL]\n"
      "addvl x25, x25, #8\n"
      "19:"  // Width 2: Output done
      "b 36f\n"
      "20:"  // Width 3
      "mov x20, #0x2\n"
      "mov x23, %x[A_ptr]\n"
      "lsl x21, %x[K], #0x2\n"
      "msub x20, x9, x20, %x[N]\n"
      "mov x22, %x[K]\n"
      ".inst 0xf8b54af8  // rprfm pldmany, x21, [x23]\n"
      ".inst 0x25b467f0  // whilelt p8.s, XZR, x20, VLx4\n"
      "cbz x24, 21f\n"
      ".inst 0xa040c700  // ld1w { z0.s-z3.s }, pn9.b/Z, [x24]\n"
      ".inst 0xa041c718  // ld1w { z24.s-z27.s }, pn9.b/Z, [x24, #0x4, MUL VL]\n"
      ".inst 0xa042c71c  // ld1w { z28.s-z31.s }, pn9.b/Z, [x24, #0x8, MUL VL]\n"
      ".inst 0xc0044c00  // mova za.d[x10, #0], { z0.d-z3.d }\n"
      ".inst 0xc0044f01  // mova za.d[x10, #1], { z24.d-z27.d }\n"
      ".inst 0xc0044f82  // mova za.d[x10, #2], { z28.d-z31.d }\n"
      "b 22f\n"
      "21:"  // Width 3: no bias
      ".inst 0xc00800ff  // zero { zad0, zad1, zad2, zad3, zad4, zad5, zad6, zad7 }\n"
      "22:"  // Width 3: setup done
      "cmp x22, #0x8\n"
      "ble 24f\n"
      "23:"  // Width 3: Multiply loop: Main loop head
      "whilelt p1.s, XZR, x22\n"
      "whilelt p0.s, x28, x22\n"
      ".inst 0xa040a74d  // ldnt1h { z12.h-z15.h }, pn9.b/Z, [x26]\n"
      "ld1rqw { z3.s }, p1/Z, [x23]\n"
      "sub x22, x22, #0x8\n"
      "ld1rqw { z0.s }, p0/Z, [x23, #16]\n"
      "cmp x22, #0x8\n"
      "add x23, x23, #0x20\n"
      ".inst 0xa041a759  // ldnt1h { z24.h-z27.h }, pn9.b/Z, [x26, #0x4, MUL VL]\n"
      ".inst 0x658aa863  // bfcvt z3.h, p2/M, z3.s\n"
      ".inst 0xa042a745  // ldnt1h { z4.h-z7.h }, pn9.b/Z, [x26, #0x8, MUL VL]\n"
      "addvl x26, x26, #16\n"
      ".inst 0x658aa800  // bfcvt z0.h, p2/M, z0.s\n"
      ".inst 0xa040a751  // ldnt1h { z16.h-z19.h }, pn9.b/Z, [x26]\n"
      ".inst 0xa041a75d  // ldnt1h { z28.h-z31.h }, pn9.b/Z, [x26, #0x4, MUL VL]\n"
      ".inst 0xa042a749  // ldnt1h { z8.h-z11.h }, pn9.b/Z, [x26, #0x8, MUL VL]\n"
      "addvl x26, x26, #16\n"
      "uzp1 z3.h, z3.h, z3.h\n"
      ".inst 0xa040a755  // ldnt1h { z20.h-z23.h }, pn9.b/Z, [x26]\n"
      "uzp1 z0.h, z0.h, z0.h\n"
      "trn1 z3.d, z3.d, z0.d\n"
      ".inst 0xc153d198  // bfdot za.s[x10, 0], { z12.h-z15.h }, z3.h[0]\n"
      ".inst 0xa041a74d  // ldnt1h { z12.h-z15.h }, pn9.b/Z, [x26, #0x4, MUL VL]\n"
      ".inst 0xc153d319  // bfdot za.s[x10, 1], { z24.h-z27.h }, z3.h[0]\n"
      ".inst 0xa042a759  // ldnt1h { z24.h-z27.h }, pn9.b/Z, [x26, #0x8, MUL VL]\n"
      "addvl x26, x26, #16\n"
      ".inst 0xc153d09a  // bfdot za.s[x10, 2], { z4.h-z7.h }, z3.h[0]\n"
      ".inst 0xa040a745  // ldnt1h { z4.h-z7.h }, pn9.b/Z, [x26]\n"
      ".inst 0xc153d618  // bfdot za.s[x10, 0], { z16.h-z19.h }, z3.h[1]\n"
      ".inst 0xa041a751  // ldnt1h { z16.h-z19.h }, pn9.b/Z, [x26, #0x4, MUL VL]\n"
      ".inst 0xc153d799  // bfdot za.s[x10, 1], { z28.h-z31.h }, z3.h[1]\n"
      ".inst 0xa042a75d  // ldnt1h { z28.h-z31.h }, pn9.b/Z, [x26, #0x8, MUL VL]\n"
      "addvl x26, x26, #16\n"
      ".inst 0xc153d51a  // bfdot za.s[x10, 2], { z8.h-z11.h }, z3.h[1]\n"
      ".inst 0xc153da98  // bfdot za.s[x10, 0], { z20.h-z23.h }, z3.h[2]\n"
      ".inst 0xc153d999  // bfdot za.s[x10, 1], { z12.h-z15.h }, z3.h[2]\n"
      ".inst 0xc153db1a  // bfdot za.s[x10, 2], { z24.h-z27.h }, z3.h[2]\n"
      ".inst 0xc153dc98  // bfdot za.s[x10, 0], { z4.h-z7.h }, z3.h[3]\n"
      ".inst 0xc153de19  // bfdot za.s[x10, 1], { z16.h-z19.h }, z3.h[3]\n"
      ".inst 0xc153df9a  // bfdot za.s[x10, 2], { z28.h-z31.h }, z3.h[3]\n"
      "bgt 23b\n"
      "24:"  // Width 3: Multiply loop: Single iteration only
      "whilelt p1.s, XZR, x22\n"
      "whilelt p0.s, x28, x22\n"
      ".inst 0xa040a745  // ldnt1h { z4.h-z7.h }, pn9.b/Z, [x26]\n"
      "ld1rqw { z3.s }, p1/Z, [x23]\n"
      "subs x22, x22, #0x2\n"
      "ld1rqw { z20.s }, p0/Z, [x23, #16]\n"
      "add x23, x23, #0x20\n"
      ".inst 0xa041a74d  // ldnt1h { z12.h-z15.h }, pn9.b/Z, [x26, #0x4, MUL VL]\n"
      ".inst 0x658aa863  // bfcvt z3.h, p2/M, z3.s\n"
      ".inst 0xa042a759  // ldnt1h { z24.h-z27.h }, pn9.b/Z, [x26, #0x8, MUL VL]\n"
      "addvl x26, x26, #16\n"
      ".inst 0x658aaa94  // bfcvt z20.h, p2/M, z20.s\n"
      "uzp1 z3.h, z3.h, z3.h\n"
      "uzp1 z20.h, z20.h, z20.h\n"
      "trn1 z3.d, z3.d, z20.d\n"
      ".inst 0xc153d098  // bfdot za.s[x10, 0], { z4.h-z7.h }, z3.h[0]\n"
      ".inst 0xc153d199  // bfdot za.s[x10, 1], { z12.h-z15.h }, z3.h[0]\n"
      ".inst 0xc153d31a  // bfdot za.s[x10, 2], { z24.h-z27.h }, z3.h[0]\n"
      "ble 25f\n"
      ".inst 0xa040a755  // ldnt1h { z20.h-z23.h }, pn9.b/Z, [x26]\n"
      "subs x22, x22, #0x2\n"
      ".inst 0xa041a751  // ldnt1h { z16.h-z19.h }, pn9.b/Z, [x26, #0x4, MUL VL]\n"
      ".inst 0xa042a75d  // ldnt1h { z28.h-z31.h }, pn9.b/Z, [x26, #0x8, MUL VL]\n"
      "addvl x26, x26, #16\n"
      ".inst 0xc153d698  // bfdot za.s[x10, 0], { z20.h-z23.h }, z3.h[1]\n"
      ".inst 0xc153d619  // bfdot za.s[x10, 1], { z16.h-z19.h }, z3.h[1]\n"
      ".inst 0xc153d79a  // bfdot za.s[x10, 2], { z28.h-z31.h }, z3.h[1]\n"
      "ble 25f\n"
      ".inst 0xa040a75d  // ldnt1h { z28.h-z31.h }, pn9.b/Z, [x26]\n"
      "subs x22, x22, #0x2\n"
      ".inst 0xa041a749  // ldnt1h { z8.h-z11.h }, pn9.b/Z, [x26, #0x4, MUL VL]\n"
      ".inst 0xa042a74d  // ldnt1h { z12.h-z15.h }, pn9.b/Z, [x26, #0x8, MUL VL]\n"
      "addvl x26, x26, #16\n"
      ".inst 0xc153db98  // bfdot za.s[x10, 0], { z28.h-z31.h }, z3.h[2]\n"
      ".inst 0xc153d919  // bfdot za.s[x10, 1], { z8.h-z11.h }, z3.h[2]\n"
      ".inst 0xc153d99a  // bfdot za.s[x10, 2], { z12.h-z15.h }, z3.h[2]\n"
      "ble 25f\n"
      ".inst 0xa040a74d  // ldnt1h { z12.h-z15.h }, pn9.b/Z, [x26]\n"
      ".inst 0xa041a759  // ldnt1h { z24.h-z27.h }, pn9.b/Z, [x26, #0x4, MUL VL]\n"
      ".inst 0xa042a749  // ldnt1h { z8.h-z11.h }, pn9.b/Z, [x26, #0x8, MUL VL]\n"
      "addvl x26, x26, #16\n"
      ".inst 0xc153dd98  // bfdot za.s[x10, 0], { z12.h-z15.h }, z3.h[3]\n"
      ".inst 0xc153df19  // bfdot za.s[x10, 1], { z24.h-z27.h }, z3.h[3]\n"
      ".inst 0xc153dd1a  // bfdot za.s[x10, 2], { z8.h-z11.h }, z3.h[3]\n"
      "25:"  // Width 3: Multiply loop: multiply skip
      "tbz %x[flags], #1, 26f\n"
      "add x21, %x[args_ptr], %[offset_min]\n"
      "add x20, %x[args_ptr], %[offset_max]\n"
      ".inst 0xc0064c08  // mova { z8.d-z11.d }, za.d[x10, #0]\n"
      ".inst 0xc0064c20  // mova { z0.d-z3.d }, za.d[x10, #1]\n"
      "ld1rw { z17.s }, p2/Z, [x21]\n"
      ".inst 0xc0064c44  // mova { z4.d-z7.d }, za.d[x10, #2]\n"
      "ld1rw { z16.s }, p2/Z, [x20]\n"
      ".inst 0xc1b0ca28  // fclamp { z8.s-z11.s }, z17.s, z16.s\n"
      ".inst 0xc1b0ca20  // fclamp { z0.s-z3.s }, z17.s, z16.s\n"
      ".inst 0xc1b0ca24  // fclamp { z4.s-z7.s }, z17.s, z16.s\n"
      ".inst 0xa060c728  // st1w { z8.s-z11.s }, pn9.b, [x25]\n"
      ".inst 0xa061c720  // st1w { z0.s-z3.s }, pn9.b, [x25, #0x4, MUL VL]\n"
      ".inst 0xa062c324  // st1w { z4.s-z7.s }, p8, [x25, #0x8, MUL VL]\n"
      "addvl x25, x25, #12\n"
      "b 27f\n"
      "26:"  // Width 3: No activation
      ".inst 0xc0064c08  // mova { z8.d-z11.d }, za.d[x10, #0]\n"
      ".inst 0xc0064c20  // mova { z0.d-z3.d }, za.d[x10, #1]\n"
      ".inst 0xc0064c50  // mova { z16.d-z19.d }, za.d[x10, #2]\n"
      ".inst 0xa060c728  // st1w { z8.s-z11.s }, pn9.b, [x25]\n"
      ".inst 0xa061c720  // st1w { z0.s-z3.s }, pn9.b, [x25, #0x4, MUL VL]\n"
      ".inst 0xa062c330  // st1w { z16.s-z19.s }, p8, [x25, #0x8, MUL VL]\n"
      "addvl x25, x25, #12\n"
      "27:"  // Width 3: Output done
      "b 36f\n"
      "28:"  // Width 4
      "mov x20, #0x3\n"
      "mov x23, %x[A_ptr]\n"
      "lsl x21, %x[K], #0x2\n"
      "msub x20, x9, x20, %x[N]\n"
      "mov x22, %x[K]\n"
      ".inst 0xf8b54af8  // rprfm pldmany, x21, [x23]\n"
      ".inst 0x25b467f0  // whilelt p8.s, XZR, x20, VLx4\n"
      "cbz x24, 29f\n"
      ".inst 0xa040c70c  // ld1w { z12.s-z15.s }, pn9.b/Z, [x24]\n"
      ".inst 0xa041c704  // ld1w { z4.s-z7.s }, pn9.b/Z, [x24, #0x4, MUL VL]\n"
      ".inst 0xa042c714  // ld1w { z20.s-z23.s }, pn9.b/Z, [x24, #0x8, MUL VL]\n"
      ".inst 0xa043c710  // ld1w { z16.s-z19.s }, pn9.b/Z, [x24, #0xc, MUL VL]\n"
      ".inst 0xc0044d80  // mova za.d[x10, #0], { z12.d-z15.d }\n"
      "addvl x24, x24, #16\n"
      ".inst 0xc0044c81  // mova za.d[x10, #1], { z4.d-z7.d }\n"
      ".inst 0xc0044e82  // mova za.d[x10, #2], { z20.d-z23.d }\n"
      ".inst 0xc0044e03  // mova za.d[x10, #3], { z16.d-z19.d }\n"
      "b 30f\n"
      "29:"  // Width 4: no bias
      ".inst 0xc00800ff  // zero { zad0, zad1, zad2, zad3, zad4, zad5, zad6, zad7 }\n"
      "30:"  // Width 4: setup done
      "cmp x22, #0x8\n"
      "ble 32f\n"
      "31:"  // Width 4: Multiply loop: Main loop head
      "whilelt p1.s, XZR, x22\n"
      "whilelt p0.s, x28, x22\n"
      ".inst 0xa040a751  // ldnt1h { z16.h-z19.h }, pn9.b/Z, [x26]\n"
      "ld1rqw { z7.s }, p1/Z, [x23]\n"
      "sub x22, x22, #0x8\n"
      "ld1rqw { z4.s }, p0/Z, [x23, #16]\n"
      "cmp x22, #0x8\n"
      "add x23, x23, #0x20\n"
      ".inst 0xa041a74d  // ldnt1h { z12.h-z15.h }, pn9.b/Z, [x26, #0x4, MUL VL]\n"
      ".inst 0x658aa8e7  // bfcvt z7.h, p2/M, z7.s\n"
      ".inst 0xa042a75d  // ldnt1h { z28.h-z31.h }, pn9.b/Z, [x26, #0x8, MUL VL]\n"
      ".inst 0x658aa884  // bfcvt z4.h, p2/M, z4.s\n"
      ".inst 0xa043a759  // ldnt1h { z24.h-z27.h }, pn9.b/Z, [x26, #0xc, MUL VL]\n"
      "addvl x26, x26, #16\n"
      ".inst 0xa040a755  // ldnt1h { z20.h-z23.h }, pn9.b/Z, [x26]\n"
      ".inst 0xa041a749  // ldnt1h { z8.h-z11.h }, pn9.b/Z, [x26, #0x4, MUL VL]\n"
      "uzp1 z7.h, z7.h, z7.h\n"
      ".inst 0xa042a741  // ldnt1h { z0.h-z3.h }, pn9.b/Z, [x26, #0x8, MUL VL]\n"
      "uzp1 z4.h, z4.h, z4.h\n"
      "trn1 z7.d, z7.d, z4.d\n"
      ".inst 0xc157d218  // bfdot za.s[x10, 0], { z16.h-z19.h }, z7.h[0]\n"
      ".inst 0xa043a751  // ldnt1h { z16.h-z19.h }, pn9.b/Z, [x26, #0xc, MUL VL]\n"
      "addvl x26, x26, #16\n"
      ".inst 0xc157d199  // bfdot za.s[x10, 1], { z12.h-z15.h }, z7.h[0]\n"
      ".inst 0xa040a74d  // ldnt1h { z12.h-z15.h }, pn9.b/Z, [x26]\n"
      ".inst 0xc157d39a  // bfdot za.s[x10, 2], { z28.h-z31.h }, z7.h[0]\n"
      ".inst 0xa041a75d  // ldnt1h { z28.h-z31.h }, pn9.b/Z, [x26, #0x4, MUL VL]\n"
      ".inst 0xc157d31b  // bfdot za.s[x10, 3], { z24.h-z27.h }, z7.h[0]\n"
      ".inst 0xa042a759  // ldnt1h { z24.h-z27.h }, pn9.b/Z, [x26, #0x8, MUL VL]\n"
      ".inst 0xc157d698  // bfdot za.s[x10, 0], { z20.h-z23.h }, z7.h[1]\n"
      ".inst 0xa043a755  // ldnt1h { z20.h-z23.h }, pn9.b/Z, [x26, #0xc, MUL VL]\n"
      "addvl x26, x26, #16\n"
      ".inst 0xc157d519  // bfdot za.s[x10, 1], { z8.h-z11.h }, z7.h[1]\n"
      ".inst 0xa040a749  // ldnt1h { z8.h-z11.h }, pn9.b/Z, [x26]\n"
      ".inst 0xc157d41a  // bfdot za.s[x10, 2], { z0.h-z3.h }, z7.h[1]\n"
      ".inst 0xa041a741  // ldnt1h { z0.h-z3.h }, pn9.b/Z, [x26, #0x4, MUL VL]\n"
      ".inst 0xc157d61b  // bfdot za.s[x10, 3], { z16.h-z19.h }, z7.h[1]\n"
      ".inst 0xa042a751  // ldnt1h { z16.h-z19.h }, pn9.b/Z, [x26, #0x8, MUL VL]\n"
      ".inst 0xc157d998  // bfdot za.s[x10, 0], { z12.h-z15.h }, z7.h[2]\n"
      ".inst 0xa043a74d  // ldnt1h { z12.h-z15.h }, pn9.b/Z, [x26, #0xc, MUL VL]\n"
      "addvl x26, x26, #16\n"
      ".inst 0xc157db99  // bfdot za.s[x10, 1], { z28.h-z31.h }, z7.h[2]\n"
      ".inst 0xc157db1a  // bfdot za.s[x10, 2], { z24.h-z27.h }, z7.h[2]\n"
      ".inst 0xc157da9b  // bfdot za.s[x10, 3], { z20.h-z23.h }, z7.h[2]\n"
      ".inst 0xc157dd18  // bfdot za.s[x10, 0], { z8.h-z11.h }, z7.h[3]\n"
      ".inst 0xc157dc19  // bfdot za.s[x10, 1], { z0.h-z3.h }, z7.h[3]\n"
      ".inst 0xc157de1a  // bfdot za.s[x10, 2], { z16.h-z19.h }, z7.h[3]\n"
      ".inst 0xc157dd9b  // bfdot za.s[x10, 3], { z12.h-z15.h }, z7.h[3]\n"
      "bgt 31b\n"
      "32:"  // Width 4: Multiply loop: Single iteration only
      "whilelt p1.s, XZR, x22\n"
      "whilelt p0.s, x28, x22\n"
      ".inst 0xa040a759  // ldnt1h { z24.h-z27.h }, pn9.b/Z, [x26]\n"
      "ld1rqw { z3.s }, p1/Z, [x23]\n"
      "subs x22, x22, #0x2\n"
      "ld1rqw { z16.s }, p0/Z, [x23, #16]\n"
      "add x23, x23, #0x20\n"
      ".inst 0xa041a75d  // ldnt1h { z28.h-z31.h }, pn9.b/Z, [x26, #0x4, MUL VL]\n"
      ".inst 0x658aa863  // bfcvt z3.h, p2/M, z3.s\n"
      ".inst 0xa042a749  // ldnt1h { z8.h-z11.h }, pn9.b/Z, [x26, #0x8, MUL VL]\n"
      ".inst 0x658aaa10  // bfcvt z16.h, p2/M, z16.s\n"
      ".inst 0xa043a74d  // ldnt1h { z12.h-z15.h }, pn9.b/Z, [x26, #0xc, MUL VL]\n"
      "addvl x26, x26, #16\n"
      "uzp1 z3.h, z3.h, z3.h\n"
      "uzp1 z16.h, z16.h, z16.h\n"
      "trn1 z3.d, z3.d, z16.d\n"
      ".inst 0xc153d318  // bfdot za.s[x10, 0], { z24.h-z27.h }, z3.h[0]\n"
      ".inst 0xc153d399  // bfdot za.s[x10, 1], { z28.h-z31.h }, z3.h[0]\n"
      ".inst 0xc153d11a  // bfdot za.s[x10, 2], { z8.h-z11.h }, z3.h[0]\n"
      ".inst 0xc153d19b  // bfdot za.s[x10, 3], { z12.h-z15.h }, z3.h[0]\n"
      "ble 33f\n"
      ".inst 0xa040a74d  // ldnt1h { z12.h-z15.h }, pn9.b/Z, [x26]\n"
      "subs x22, x22, #0x2\n"
      ".inst 0xa041a75d  // ldnt1h { z28.h-z31.h }, pn9.b/Z, [x26, #0x4, MUL VL]\n"
      ".inst 0xa042a749  // ldnt1h { z8.h-z11.h }, pn9.b/Z, [x26, #0x8, MUL VL]\n"
      ".inst 0xa043a759  // ldnt1h { z24.h-z27.h }, pn9.b/Z, [x26, #0xc, MUL VL]\n"
      ".inst 0xc153d598  // bfdot za.s[x10, 0], { z12.h-z15.h }, z3.h[1]\n"
      "addvl x26, x26, #16\n"
      ".inst 0xc153d799  // bfdot za.s[x10, 1], { z28.h-z31.h }, z3.h[1]\n"
      ".inst 0xc153d51a  // bfdot za.s[x10, 2], { z8.h-z11.h }, z3.h[1]\n"
      ".inst 0xc153d71b  // bfdot za.s[x10, 3], { z24.h-z27.h }, z3.h[1]\n"
      "ble 33f\n"
      ".inst 0xa040a759  // ldnt1h { z24.h-z27.h }, pn9.b/Z, [x26]\n"
      "subs x22, x22, #0x2\n"
      ".inst 0xa041a74d  // ldnt1h { z12.h-z15.h }, pn9.b/Z, [x26, #0x4, MUL VL]\n"
      ".inst 0xa042a75d  // ldnt1h { z28.h-z31.h }, pn9.b/Z, [x26, #0x8, MUL VL]\n"
      ".inst 0xa043a751  // ldnt1h { z16.h-z19.h }, pn9.b/Z, [x26, #0xc, MUL VL]\n"
      ".inst 0xc153db18  // bfdot za.s[x10, 0], { z24.h-z27.h }, z3.h[2]\n"
      "addvl x26, x26, #16\n"
      ".inst 0xc153d999  // bfdot za.s[x10, 1], { z12.h-z15.h }, z3.h[2]\n"
      ".inst 0xc153db9a  // bfdot za.s[x10, 2], { z28.h-z31.h }, z3.h[2]\n"
      ".inst 0xc153da1b  // bfdot za.s[x10, 3], { z16.h-z19.h }, z3.h[2]\n"
      "ble 33f\n"
      ".inst 0xa040a749  // ldnt1h { z8.h-z11.h }, pn9.b/Z, [x26]\n"
      ".inst 0xa041a759  // ldnt1h { z24.h-z27.h }, pn9.b/Z, [x26, #0x4, MUL VL]\n"
      ".inst 0xa042a755  // ldnt1h { z20.h-z23.h }, pn9.b/Z, [x26, #0x8, MUL VL]\n"
      ".inst 0xa043a751  // ldnt1h { z16.h-z19.h }, pn9.b/Z, [x26, #0xc, MUL VL]\n"
      ".inst 0xc153dd18  // bfdot za.s[x10, 0], { z8.h-z11.h }, z3.h[3]\n"
      "addvl x26, x26, #16\n"
      ".inst 0xc153df19  // bfdot za.s[x10, 1], { z24.h-z27.h }, z3.h[3]\n"
      ".inst 0xc153de9a  // bfdot za.s[x10, 2], { z20.h-z23.h }, z3.h[3]\n"
      ".inst 0xc153de1b  // bfdot za.s[x10, 3], { z16.h-z19.h }, z3.h[3]\n"
      "33:"  // Width 4: Multiply loop: multiply skip
      "tbz %x[flags], #1, 34f\n"
      "add x21, %x[args_ptr], %[offset_min]\n"
      "add x20, %x[args_ptr], %[offset_max]\n"
      ".inst 0xc0064c04  // mova { z4.d-z7.d }, za.d[x10, #0]\n"
      ".inst 0xc0064c28  // mova { z8.d-z11.d }, za.d[x10, #1]\n"
      "ld1rw { z21.s }, p2/Z, [x21]\n"
      ".inst 0xc0064c4c  // mova { z12.d-z15.d }, za.d[x10, #2]\n"
      "ld1rw { z20.s }, p2/Z, [x20]\n"
      ".inst 0xc0064c70  // mova { z16.d-z19.d }, za.d[x10, #3]\n"
      ".inst 0xc1b4caa4  // fclamp { z4.s-z7.s }, z21.s, z20.s\n"
      ".inst 0xc1b4caa8  // fclamp { z8.s-z11.s }, z21.s, z20.s\n"
      ".inst 0xc1b4caac  // fclamp { z12.s-z15.s }, z21.s, z20.s\n"
      ".inst 0xc1b4cab0  // fclamp { z16.s-z19.s }, z21.s, z20.s\n"
      ".inst 0xa060c724  // st1w { z4.s-z7.s }, pn9.b, [x25]\n"
      ".inst 0xa061c728  // st1w { z8.s-z11.s }, pn9.b, [x25, #0x4, MUL VL]\n"
      ".inst 0xa062c72c  // st1w { z12.s-z15.s }, pn9.b, [x25, #0x8, MUL VL]\n"
      ".inst 0xa063c330  // st1w { z16.s-z19.s }, p8, [x25, #0xc, MUL VL]\n"
      "addvl x25, x25, #16\n"
      "b 35f\n"
      "34:"  // Width 4: No activation
      ".inst 0xc0064c00  // mova { z0.d-z3.d }, za.d[x10, #0]\n"
      ".inst 0xc0064c24  // mova { z4.d-z7.d }, za.d[x10, #1]\n"
      ".inst 0xc0064c4c  // mova { z12.d-z15.d }, za.d[x10, #2]\n"
      ".inst 0xc0064c78  // mova { z24.d-z27.d }, za.d[x10, #3]\n"
      ".inst 0xa060c720  // st1w { z0.s-z3.s }, pn9.b, [x25]\n"
      ".inst 0xa061c724  // st1w { z4.s-z7.s }, pn9.b, [x25, #0x4, MUL VL]\n"
      ".inst 0xa062c72c  // st1w { z12.s-z15.s }, pn9.b, [x25, #0x8, MUL VL]\n"
      ".inst 0xa063c338  // st1w { z24.s-z27.s }, p8, [x25, #0xc, MUL VL]\n"
      "addvl x25, x25, #16\n"
      "35:"  // Width 4: Output done
      "subs x27, x27, #0x4\n"
      "sub %x[N], %x[N], x9, LSL #2\n"
      "bgt 4b\n"
      "36:"  // Exit
      ".inst 0xd503467f  // SMSTOP\n"
      "ptrue p8.b\n"
      : [N] "+&r" (N)
      : [A_ptr] "r" (A_ptr), [B_ptr] "r" (B_ptr), [K] "r" (K), [args_ptr] "r" (&ka), [bias] "r" (bias), [flags] "r" (flags), [offset_max] "I" (offsetof(KernelArgs, maxval)), [offset_min] "I" (offsetof(KernelArgs, minval)), [output_ptr] "r" (output_ptr)
      : "cc", "memory", "p0", "p1", "p2", "p3", "p4", "p5", "p6", "p7", "p8", "p9", "p10", "p11", "p12", "p13", "p14", "p15", "x9", "x10", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31"
    );
}

} // namespace arm_gemm

#endif  // defined(ARM_COMPUTE_ENABLE_SME2)
