/*
 * Copyright (C) 2015 Martin Willi
 *
 * Copyright (C) secunet Security Networks AG
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  See <http://www.fsf.org/copyleft/gpl.txt>.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 */

#include <test_suite.h>

#include <ip_packet.h>
#include <esp_packet.h>

static iv_gen_t *ivgen;

METHOD(aead_t, get_iv_gen, iv_gen_t*,
	aead_t *this)
{
	return ivgen;
}

METHOD(iv_gen_t, get_iv, bool,
	iv_gen_t *this, uint64_t seq, size_t size, uint8_t *buffer)
{
	if (size != 8)
	{
		return FALSE;
	}
	memcpy(buffer, "\x10\x11\x12\x13\x14\x15\x16\x17", 8);
	return TRUE;
}

METHOD(iv_gen_t, allocate_iv, bool,
	iv_gen_t *this, uint64_t seq, size_t size, chunk_t *chunk)
{
	if (size != 8)
	{
		return FALSE;
	}
	*chunk = chunk_alloc(size);
	return get_iv(this, seq, chunk->len, chunk->ptr);
}

/**
 * Appendix A draft-ietf-ipsecme-chacha20-poly1305-06
 */
START_TEST(test_chapoly)
{
	host_t *src, *dst;
	ip_packet_t *icmp;
	esp_packet_t *esp;
	esp_context_t *ctx;
	chunk_t data, exp;
	uint32_t seq = 0;

	icmp = ip_packet_create(chunk_clone(chunk_from_chars(
								0x45,0x00,0x00,0x54,0xa6,0xf2,0x00,0x00,
								0x40,0x01,0xe7,0x78,0xc6,0x33,0x64,0x05,
								0xc0,0x00,0x02,0x05,0x08,0x00,0x5b,0x7a,
								0x3a,0x08,0x00,0x00,0x55,0x3b,0xec,0x10,
								0x00,0x07,0x36,0x27,0x08,0x09,0x0a,0x0b,
								0x0c,0x0d,0x0e,0x0f,0x10,0x11,0x12,0x13,
								0x14,0x15,0x16,0x17,0x18,0x19,0x1a,0x1b,
								0x1c,0x1d,0x1e,0x1f,0x20,0x21,0x22,0x23,
								0x24,0x25,0x26,0x27,0x28,0x29,0x2a,0x2b,
								0x2c,0x2d,0x2e,0x2f,0x30,0x31,0x32,0x33,
								0x34,0x35,0x36,0x37)));
	ck_assert(icmp);

	src = host_create_from_string("203.0.113.153", 0);
	dst = host_create_from_string("203.0.113.5", 0);
	esp = esp_packet_create_from_payload(src, dst, icmp);

	ctx = esp_context_create(ENCR_CHACHA20_POLY1305, chunk_from_chars(
								0x80,0x81,0x82,0x83,0x84,0x85,0x86,0x87,
								0x88,0x89,0x8a,0x8b,0x8c,0x8d,0x8e,0x8f,
								0x90,0x91,0x92,0x93,0x94,0x95,0x96,0x97,
								0x98,0x99,0x9a,0x9b,0x9c,0x9d,0x9e,0x9f,
								0xa0,0xa1,0xa2,0xa3),
							 AUTH_UNDEFINED, chunk_empty, FALSE);
	while (seq != 4)
	{
		ck_assert(ctx->next_seqno(ctx, &seq));
	}
	INIT(ivgen,
		.get_iv = _get_iv,
		.allocate_iv = _allocate_iv,
		.destroy = (void*)free,
	);
	ctx->get_aead(ctx)->get_iv_gen = _get_iv_gen;
	ck_assert(esp->encrypt(esp, ctx, htonl(0x01020304)) == SUCCESS);

	data = esp->packet.get_data(&esp->packet);
	exp = chunk_from_chars(0x01,0x02,0x03,0x04,0x00,0x00,0x00,0x05,
						   0x10,0x11,0x12,0x13,0x14,0x15,0x16,0x17,
						   0x24,0x03,0x94,0x28,0xb9,0x7f,0x41,0x7e,
						   0x3c,0x13,0x75,0x3a,0x4f,0x05,0x08,0x7b,
						   0x67,0xc3,0x52,0xe6,0xa7,0xfa,0xb1,0xb9,
						   0x82,0xd4,0x66,0xef,0x40,0x7a,0xe5,0xc6,
						   0x14,0xee,0x80,0x99,0xd5,0x28,0x44,0xeb,
						   0x61,0xaa,0x95,0xdf,0xab,0x4c,0x02,0xf7,
						   0x2a,0xa7,0x1e,0x7c,0x4c,0x4f,0x64,0xc9,
						   0xbe,0xfe,0x2f,0xac,0xc6,0x38,0xe8,0xf3,
						   0xcb,0xec,0x16,0x3f,0xac,0x46,0x9b,0x50,
						   0x27,0x73,0xf6,0xfb,0x94,0xe6,0x64,0xda,
						   0x91,0x65,0xb8,0x28,0x29,0xf6,0x41,0xe0,
						   0x76,0xAA,0xA8,0x26,0x6B,0x7F,0xB0,0xF7,
						   0xB1,0x1B,0x36,0x99,0x07,0xE1,0xAD,0x43);
	ck_assert_msg(chunk_equals(data, exp), "got %B\nexp %B", &data, &exp);

	esp->destroy(esp);
	ctx->destroy(ctx);
	ivgen->destroy(ivgen);
}
END_TEST

Suite *chapoly_suite_create()
{
	Suite *s;
	TCase *tc;

	s = suite_create("chapoly");

	tc = tcase_create("ChaCha20Poly1305 ESP encryption");
	tcase_add_test(tc, test_chapoly);
	suite_add_tcase(s, tc);

	return s;
}
